"use client";

import { useState } from "react";
import Image from "next/image";
import Link from "next/link";
import type { SiteInfo } from "@/types/data";
import type { Translations } from "@/lib/translations";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface HeaderProps {
  readonly siteInfo: SiteInfo;
  readonly translations: Translations;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

export function Header({ siteInfo, translations, language, theme }: HeaderProps) {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const colorTheme = getColorTheme(theme);

  const navLinks = [
    { href: "/", label: translations.navigation.home },
    { href: "/about", label: translations.navigation.about },
    { href: "/contact", label: translations.navigation.contact },
  ];

  return (
    <header 
      className="sticky top-0 z-50 border-b backdrop-blur-md shadow-xl"
      style={{
        borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
        background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
        boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
      }}
    >
      {/* Subtle background gradients - theme-based */}
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
        }}
      />
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
        }}
      />
      
      {/* Animated grid pattern */}
      <div className="absolute inset-0 bg-[linear-gradient(rgba(255,255,255,0.02)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,0.02)_1px,transparent_1px)] bg-[size:30px_30px] opacity-40" />
      
      <div className="relative mx-auto flex max-w-6xl items-center justify-between px-4 py-4 sm:px-6 sm:py-5">
        <Link href="/" className="group flex items-center gap-2.5 sm:gap-3 transition-opacity hover:opacity-90">
          {/* Logo section - using PartnerCard approach */}
          <div className="relative flex-shrink-0">
            <div 
              className="relative h-12 w-12 overflow-hidden rounded-xl border-2 bg-white p-1 shadow-md sm:shadow-lg transition-all duration-500 group-hover:scale-110 group-hover:shadow-xl sm:h-16 sm:w-16 sm:rounded-3xl sm:p-3"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.6),
                boxShadow: `0 4px 6px -1px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
              }}
              onMouseEnter={(e) => {
                e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.8);
                e.currentTarget.style.boxShadow = `0 20px 25px -5px ${hexToRgba(colorTheme.buttonBadge, 0.4)}`;
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.6);
                e.currentTarget.style.boxShadow = `0 4px 6px -1px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
              }}
            >
              {/* Sparkle effect around logo - buttonBadge color */}
              <div 
                className="absolute -inset-2 rounded-xl sm:opacity-0 blur-md transition-all duration-500 group-hover:opacity-100 sm:rounded-3xl" 
                style={{
                  background: `linear-gradient(to right, ${hexToRgba(colorTheme.buttonBadge, 0.15)}, ${hexToRgba(colorTheme.buttonBadge, 0.10)}, ${hexToRgba(colorTheme.buttonBadge, 0.15)})`,
                }}
              />
              <Image
                src={siteInfo.siteLogo}
                alt={`${siteInfo.siteName} logo`}
                fill
                className="relative z-10 object-contain brightness-100 contrast-100"
                sizes="(max-width: 640px) 48px, 64px"
                priority
                fetchPriority="high"
                quality={85}
                unoptimized={!siteInfo.siteLogo.includes('logo.png')}
              />
            </div>
          </div>
          <div className="flex flex-col">
            <span className="text-lg font-bold tracking-tight text-white sm:text-xl">
              {siteInfo.siteName}
            </span>
            <span className="text-[10px] font-medium text-white/70 sm:text-xs">
              {siteInfo.tagline}
            </span>
          </div>
        </Link>
        
        {/* Mobile menu button */}
        <button
          type="button"
          onClick={() => setIsMenuOpen(!isMenuOpen)}
          className="relative flex flex-col gap-1.5 rounded-xl border p-2.5 transition-all active:scale-95 sm:hidden focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
            backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.05),
          }}
          onMouseEnter={(e) => {
            e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.1);
            e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.3);
          }}
          onMouseLeave={(e) => {
            e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.05);
            e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.2);
          }}
          onFocus={(e) => {
            e.currentTarget.style.outlineColor = colorTheme.buttonBadge;
          }}
          aria-label={translations.navigation.home}
          aria-expanded={isMenuOpen}
        >
          <span
            className={`h-0.5 w-5 origin-center rounded-full bg-white transition-all duration-300 ${
              isMenuOpen ? "translate-y-2 rotate-45" : ""
            }`}
          />
          <span
            className={`h-0.5 w-5 rounded-full bg-white transition-all duration-300 ${
              isMenuOpen ? "opacity-0" : "opacity-100"
            }`}
          />
          <span
            className={`h-0.5 w-5 origin-center rounded-full bg-white transition-all duration-300 ${
              isMenuOpen ? "-translate-y-2 -rotate-45" : ""
            }`}
          />
        </button>

        {/* Desktop navigation */}
        <nav className="hidden items-center gap-2 text-sm font-semibold sm:flex">
          {navLinks.map((link) => (
            <Link
              key={link.href}
              href={link.href}
              className="relative rounded-xl px-4 py-2.5 text-white/90 transition-all duration-200 hover:text-white active:scale-95 sm:px-5 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2"
              style={{
                outlineColor: colorTheme.buttonBadge,
              }}
              onMouseEnter={(e) => {
                e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.2);
                e.currentTarget.style.boxShadow = `0 10px 15px -3px ${hexToRgba(colorTheme.buttonBadge, 0.3)}`;
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.backgroundColor = 'transparent';
                e.currentTarget.style.boxShadow = 'none';
              }}
            >
              {link.label}
            </Link>
          ))}
        </nav>
      </div>

      {/* Mobile navigation */}
      {isMenuOpen && (
        <nav 
          className="relative border-t backdrop-blur-md sm:hidden"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
            background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
            boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
          }}
        >
          {/* Subtle background gradients */}
          <div 
            className="absolute inset-0" 
            style={{
              background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
            }}
          />
          <div 
            className="absolute inset-0" 
            style={{
              background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
            }}
          />
          <div className="relative mx-auto flex max-w-6xl flex-col gap-1 px-4 py-3">
            {navLinks.map((link) => (
              <Link
                key={link.href}
                href={link.href}
                onClick={() => setIsMenuOpen(false)}
                className="rounded-xl px-4 py-3 text-sm font-semibold text-white transition-all duration-200 active:scale-[0.98] focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2"
                style={{
                  outlineColor: colorTheme.buttonBadge,
                }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.2);
                  e.currentTarget.style.boxShadow = `0 10px 15px -3px ${hexToRgba(colorTheme.buttonBadge, 0.3)}`;
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.backgroundColor = 'transparent';
                  e.currentTarget.style.boxShadow = 'none';
                }}
              >
                {link.label}
              </Link>
            ))}
          </div>
        </nav>
      )}
    </header>
  );
}
