import type { Metadata } from "next";
import { createMetadata } from "@/lib/metadata";
import { ContactForm } from "@/components/ContactForm";
import { ContactAddressItem } from "@/components/ContactAddressItem";
import { getSiteInfo } from "@/lib/data";
import { getLanguage } from "@/lib/language";
import { getTheme } from "@/lib/theme";
import { getTranslations } from "@/lib/translations";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

const descriptionsByLang = {
  en: "Get in touch with the {siteName} team for partnership inquiries, compliance questions, or support via email, phone, or contact form.",
  pt: "Entre em contacto com a equipa do {siteName} para consultas de parceria, questões de conformidade ou suporte via email, telefone ou formulário de contacto.",
  dk: "Kom i kontakt med {siteName}-teamet for partnerskabsforespørgsler, overensstemmelsesspørgsmål eller support via email, telefon eller kontaktformular.",
  fr: "Entrez en contact avec l'équipe {siteName} pour les demandes de partenariat, les questions de conformité ou le support par email, téléphone ou formulaire de contact.",
  nl: "Neem contact op met het {siteName} team voor partnerschapsinformatie, compliance vragen of ondersteuning via e-mail, telefoon of contactformulier.",
  sv: "Kontakta {siteName}-teamet för partnerskapsförfrågningar, efterlevnadsfrågor eller support via e-post, telefon eller kontaktformulär.",
  es: "Ponerse en contacto con el equipo de {siteName} para consultas de asociación, preguntas de cumplimiento o soporte por correo electrónico, teléfono o formulario de contacto.",
} as const;

export async function generateMetadata(): Promise<Metadata> {
  const siteInfo = getSiteInfo();
  const language = await getLanguage();
  const description = descriptionsByLang[language].replace("{siteName}", siteInfo.siteName);
  
  return createMetadata({
    title: `Contact ${siteInfo.siteName}`,
    description,
    path: "/contact",
    language,
  });
}

export default async function ContactPage() {
  const siteInfo = getSiteInfo();
  const language = await getLanguage();
  const theme = await getTheme();
  const translations = getTranslations(language);
  const colorTheme = getColorTheme(theme);
  
  return (
    <div className="space-y-6 sm:space-y-8">
      <section 
        className="relative overflow-hidden rounded-2xl border px-6 py-8 text-white shadow-2xl sm:rounded-3xl sm:px-8 sm:py-10"
        style={{
          borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
          background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
          boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
        }}
      >
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
          }}
        />
        <div 
          className="absolute inset-0" 
          style={{
            background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
          }}
        />
        <div className="relative space-y-4 text-center sm:space-y-5">
          <span 
            className="inline-flex items-center gap-2 rounded-full border px-3 py-1.5 text-[10px] font-semibold uppercase tracking-[0.15em] backdrop-blur-sm sm:px-4 sm:py-2 sm:text-xs"
            style={{
              borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
              backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.1),
              color: hexToRgba(colorTheme.buttonBadge, 0.9),
            }}
          >
            <span className="relative flex h-1.5 w-1.5 sm:h-2 sm:w-2">
              <span 
                className="absolute inline-flex h-full w-full animate-ping rounded-full opacity-75" 
                style={{ backgroundColor: colorTheme.buttonBadge }}
              />
              <span 
                className="relative inline-flex h-full w-full rounded-full" 
                style={{ backgroundColor: colorTheme.buttonBadge }}
              />
            </span>
            {translations.contact.badge}
          </span>
          <h1 className="text-3xl font-bold leading-[1.1] tracking-tight text-white sm:text-5xl lg:text-6xl">
            {translations.contact.title}
          </h1>
          <p className="mx-auto max-w-3xl text-sm leading-relaxed text-white/90 sm:text-lg">
            {translations.contact.subtitle}
          </p>
        </div>
      </section>

      <div className="grid gap-6 sm:gap-8 lg:grid-cols-[1.2fr_1fr]">
        <section 
          className="group relative space-y-4 overflow-hidden rounded-2xl border p-5 shadow-xl backdrop-blur-sm transition-all duration-500 ease-out sm:rounded-3xl sm:space-y-5 sm:p-8"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
            background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
            boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
          }}
        >
          <div 
            className="absolute inset-0" 
            style={{
              background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
            }}
          />
          <div 
            className="absolute inset-0" 
            style={{
              background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
            }}
          />
          <div className="relative z-10 space-y-2 sm:space-y-3">
            <div className="flex items-center gap-2.5 sm:gap-3">
              <div 
                className="flex h-10 w-10 shrink-0 items-center justify-center rounded-xl text-white shadow-lg transition-transform duration-300 group-hover:scale-110 sm:h-12 sm:w-12"
                style={{
                  background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                  boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                }}
              >
                <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                  <path strokeLinecap="round" strokeLinejoin="round" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                </svg>
              </div>
              <div>
                <h2 className="text-lg font-bold tracking-tight text-white sm:text-2xl">
                  {translations.contact.formTitle}
                </h2>
                <p className="mt-0.5 text-xs leading-relaxed text-white/80 sm:text-sm">
                  {translations.contact.formDescription}
                </p>
              </div>
            </div>
          </div>
          <div className="relative z-10">
            <ContactForm language={language} theme={theme} />
          </div>
        </section>
        <aside
          className="relative space-y-4 overflow-hidden rounded-2xl border p-5 shadow-xl backdrop-blur-sm sm:rounded-3xl sm:space-y-5 sm:p-8"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
            background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
            boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
          }}
        >
          <div 
            className="absolute inset-0" 
            style={{
              background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
            }}
          />
          <div className="relative z-10 flex items-center gap-2.5 sm:gap-3">
            <div 
              className="flex h-10 w-10 shrink-0 items-center justify-center rounded-xl text-white shadow-lg transition-transform duration-300 group-hover:scale-110 sm:h-12 sm:w-12"
              style={{
                background: `linear-gradient(to bottom right, ${colorTheme.welcomeBonus}, ${colorTheme.welcomeBonus})`,
                boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.welcomeBonus, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
              }}
            >
              <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
              </svg>
            </div>
            <h2 className="text-lg font-bold text-white sm:text-xl">{translations.contact.contactDetails}</h2>
          </div>
          <ul className="relative z-10 space-y-3 sm:space-y-3.5">
            <li 
              className="group relative flex flex-col overflow-hidden rounded-xl border p-4 shadow-lg transition-all duration-300 ease-out hover:-translate-y-1 sm:rounded-2xl sm:p-5"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
                background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
                boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)}, ${hexToRgba(colorTheme.buttonBadge, 0)})`,
                }}
              />
              <div className="relative z-10 mb-2.5 flex items-center gap-2.5 sm:mb-3 sm:gap-3">
                <div 
                  className="flex h-9 w-9 shrink-0 items-center justify-center rounded-lg text-white shadow-md transition-transform duration-300 group-hover:scale-110 sm:h-10 sm:w-10"
                  style={{
                    background: `linear-gradient(to bottom right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
                    boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.buttonBadge, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                  }}
                >
                  <svg className="h-4 w-4 sm:h-5 sm:w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                  </svg>
                </div>
                <span className="text-[10px] font-bold uppercase tracking-wider text-white/70 sm:text-xs">
                  {translations.contact.email}
                </span>
              </div>
              <a
                href={`mailto:${siteInfo.email}`}
                className="relative z-10 text-sm font-bold text-white transition-colors hover:text-blue-300 hover:underline sm:text-base"
              >
                {siteInfo.email}
              </a>
            </li>
            <li 
              className="group relative flex flex-col overflow-hidden rounded-xl border p-4 shadow-lg transition-all duration-300 ease-out hover:-translate-y-1 sm:rounded-2xl sm:p-5"
              style={{
                borderColor: hexToRgba(colorTheme.welcomeBonus, 0.25),
                background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
                boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.welcomeBonus, 0.1)}`,
              }}
            >
              <div 
                className="absolute inset-0 transition-all duration-300" 
                style={{
                  background: `linear-gradient(to bottom right, ${hexToRgba(colorTheme.welcomeBonus, 0)}, ${hexToRgba(colorTheme.welcomeBonus, 0)}, ${hexToRgba(colorTheme.welcomeBonus, 0)})`,
                }}
              />
              <div className="relative z-10 mb-2.5 flex items-center gap-2.5 sm:mb-3 sm:gap-3">
                <div 
                  className="flex h-9 w-9 shrink-0 items-center justify-center rounded-lg text-white shadow-md transition-transform duration-300 group-hover:scale-110 sm:h-10 sm:w-10"
                  style={{
                    background: `linear-gradient(to bottom right, ${colorTheme.welcomeBonus}, ${colorTheme.welcomeBonus})`,
                    boxShadow: `0 4px 12px -2px ${hexToRgba(colorTheme.welcomeBonus, 0.4)}, inset 0 1px 0 rgba(255, 255, 255, 0.1)`,
                  }}
                >
                  <svg className="h-4 w-4 sm:h-5 sm:w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2.5}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                  </svg>
                </div>
                <span className="text-[10px] font-bold uppercase tracking-wider text-white/70 sm:text-xs">
                  {translations.contact.phone}
                </span>
              </div>
              <a
                href={`tel:${siteInfo.phone.replace(/\s+/g, "")}`}
                className="relative z-10 text-sm font-bold text-white transition-colors hover:text-blue-300 hover:underline sm:text-base"
              >
                {siteInfo.phone}
              </a>
            </li>
            <ContactAddressItem 
              address={siteInfo.address}
              label={translations.contact.address}
              theme={theme}
            />
          </ul>
        </aside>
      </div>
    </div>
  );
}
