"use client";

import { useState } from "react";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface ContactFormProps {
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

export function ContactForm({ language, theme }: ContactFormProps) {
  const colorTheme = getColorTheme(theme);
  const [status, setStatus] = useState<"idle" | "success">("idle");

  return (
    <form
      className="space-y-6"
      onSubmit={(event) => {
        event.preventDefault();
        setStatus("success");
      }}
    >
      <div className="space-y-2">
        <label htmlFor="name" className="block text-sm font-semibold text-white/90">
          Nome
        </label>
        <input
          id="name"
          name="name"
          type="text"
          required
          autoComplete="name"
          className="w-full rounded-xl border backdrop-blur-sm px-5 py-3.5 text-sm text-white shadow-sm transition-all duration-200 placeholder:text-white/50 focus:outline-none focus:ring-2 hover:border-white/30"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
            backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.1),
          }}
          onFocus={(e) => {
            e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.4);
            e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.15);
            e.currentTarget.style.boxShadow = `0 0 0 2px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
          }}
          onBlur={(e) => {
            e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.2);
            e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.1);
            e.currentTarget.style.boxShadow = 'none';
          }}
          placeholder="João Silva"
        />
      </div>
      <div className="space-y-2">
        <label htmlFor="email" className="block text-sm font-semibold text-white/90">
          Email
        </label>
        <input
          id="email"
          name="email"
          type="email"
          required
          autoComplete="email"
          className="w-full rounded-xl border backdrop-blur-sm px-5 py-3.5 text-sm text-white shadow-sm transition-all duration-200 placeholder:text-white/50 focus:outline-none focus:ring-2 hover:border-white/30"
          style={{
            borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
            backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.1),
          }}
          onFocus={(e) => {
            e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.4);
            e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.15);
            e.currentTarget.style.boxShadow = `0 0 0 2px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
          }}
          onBlur={(e) => {
            e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.2);
            e.currentTarget.style.backgroundColor = hexToRgba(colorTheme.buttonBadge, 0.1);
            e.currentTarget.style.boxShadow = 'none';
          }}
          placeholder="voce@empresa.com"
        />
      </div>
      <div className="space-y-2">
        <label htmlFor="message" className="block text-sm font-semibold text-white/90">
          Mensagem
        </label>
        <textarea
          id="message"
          name="message"
          rows={5}
          required
          className="w-full resize-none rounded-xl border border-white/20 bg-white/10 backdrop-blur-sm px-5 py-3.5 text-sm text-white shadow-sm transition-all duration-200 placeholder:text-white/50 focus:border-white/40 focus:bg-white/15 focus:outline-none focus:ring-2 focus:ring-white/20 hover:border-white/30"
          placeholder="Dê-nos uma breve visão geral dos seus objetivos de campanha."
        />
      </div>
      <button
        type="submit"
        className="group/btn relative inline-flex w-full items-center justify-center gap-2 overflow-hidden rounded-xl px-6 py-4 text-sm font-semibold text-white shadow-lg transition-all duration-300 hover:scale-[1.02] hover:shadow-xl focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 active:scale-[0.98]"
        style={{
          background: `linear-gradient(to right, ${colorTheme.buttonBadge}, ${colorTheme.buttonBadge})`,
          boxShadow: `0 10px 15px -3px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
        }}
        onFocus={(e) => {
          e.currentTarget.style.outlineColor = colorTheme.buttonBadge;
        }}
        onMouseEnter={(e) => {
          e.currentTarget.style.boxShadow = `0 20px 25px -5px ${hexToRgba(colorTheme.buttonBadge, 0.3)}`;
        }}
        onMouseLeave={(e) => {
          e.currentTarget.style.boxShadow = `0 10px 15px -3px ${hexToRgba(colorTheme.buttonBadge, 0.2)}`;
        }}
      >
        <span className="relative z-10">Enviar pedido</span>
        <span 
          className="absolute inset-0 opacity-0 transition-opacity duration-300 group-hover/btn:opacity-100"
          style={{
            background: `linear-gradient(to right, ${hexToRgba(colorTheme.buttonBadge, 0.8)}, ${hexToRgba(colorTheme.buttonBadge, 0.7)}, ${hexToRgba(colorTheme.buttonBadge, 0.8)})`,
          }}
        />
      </button>
      {status === "success" && (
        <div 
          className="rounded-xl border backdrop-blur-sm p-4 shadow-sm"
          style={{
            borderColor: hexToRgba(colorTheme.welcomeBonus, 0.3),
            backgroundColor: hexToRgba(colorTheme.welcomeBonus, 0.1),
          }}
        >
          <p 
            className="text-sm font-semibold"
            style={{ color: hexToRgba(colorTheme.welcomeBonus, 0.9) }}
          >
            Obrigado por entrar em contacto! Um membro da equipa entrará em contacto consigo no prazo de 24 horas.
          </p>
        </div>
      )}
    </form>
  );
}
