"use client";

import { useState } from "react";
import type { Translations } from "@/lib/translations";
import type { LanguageCode } from "@/types/language";
import type { ThemeId } from "@/types/theme";
import { getColorTheme, hexToRgba } from "@/lib/color-theme";

interface FAQSectionProps {
  readonly translations: Translations;
  readonly language: LanguageCode;
  readonly theme: ThemeId;
}

export function FAQSection({ translations, language, theme }: FAQSectionProps) {
  const [openIndex, setOpenIndex] = useState<number | null>(null);
  const colorTheme = getColorTheme(theme);

  const toggleQuestion = (index: number) => {
    setOpenIndex(openIndex === index ? null : index);
  };

  return (
    <section 
      className="relative space-y-5 overflow-hidden rounded-2xl border p-5 shadow-xl backdrop-blur-sm sm:space-y-6 sm:rounded-3xl sm:p-8"
      style={{
        borderColor: hexToRgba(colorTheme.buttonBadge, 0.2),
        background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.98), ${hexToRgba(colorTheme.cardBackground, 0.95)}, rgba(0, 0, 0, 0.98))`,
        boxShadow: `0 20px 50px -10px rgba(0, 0, 0, 0.8), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
      }}
    >
      {/* Subtle accent gradients - theme-based */}
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 30% 20%, ${hexToRgba(colorTheme.buttonBadge, 0.06)}, transparent 60%)`,
        }}
      />
      <div 
        className="absolute inset-0" 
        style={{
          background: `radial-gradient(circle at 70% 80%, ${hexToRgba(colorTheme.welcomeBonus, 0.04)}, transparent 60%)`,
        }}
      />
      
      {/* Header section */}
      <div className="relative space-y-2 text-center sm:space-y-3">
        <div 
          className="inline-flex items-center gap-2 rounded-full px-3 py-1 backdrop-blur-sm sm:px-4 sm:py-1.5"
          style={{
            border: `1px solid ${hexToRgba(colorTheme.buttonBadge, 0.2)}`,
            backgroundColor: hexToRgba(colorTheme.buttonBadge, 0.05),
          }}
        >
          <svg 
            className="h-3 w-3 sm:h-4 sm:w-4" 
            fill="none" 
            viewBox="0 0 24 24" 
            stroke="currentColor" 
            strokeWidth={2.5}
            style={{ color: colorTheme.buttonBadge }}
          >
            <path strokeLinecap="round" strokeLinejoin="round" d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
          <span 
            className="text-[10px] font-semibold uppercase tracking-wider sm:text-xs"
            style={{ color: hexToRgba(colorTheme.buttonBadge, 0.9) }}
          >
            {translations.faq.badge}
          </span>
        </div>
        <h2 className="text-2xl font-bold tracking-tight text-white sm:text-3xl lg:text-4xl">
          {translations.faq.title}
        </h2>
        <p className="mx-auto max-w-2xl text-sm leading-relaxed text-slate-300/90 sm:text-base sm:text-white/80">
          {translations.faq.subtitle}
        </p>
      </div>
      
      {/* FAQ Items */}
      <div className="relative space-y-2.5 sm:space-y-3">
        {translations.faq.items.map((item, index) => {
          const isOpen = openIndex === index;
          return (
            <div
              key={index}
              className="group overflow-hidden rounded-xl border transition-all duration-300"
              style={{
                borderColor: hexToRgba(colorTheme.buttonBadge, 0.25),
                background: `linear-gradient(to bottom right, rgba(0, 0, 0, 0.9), ${hexToRgba(colorTheme.cardBackground, 0.85)}, rgba(0, 0, 0, 0.9))`,
                boxShadow: `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`,
              }}
              onMouseEnter={(e) => {
                e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.35);
                e.currentTarget.style.boxShadow = `0 10px 25px -5px rgba(0, 0, 0, 0.6), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.15)}`;
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.borderColor = hexToRgba(colorTheme.buttonBadge, 0.25);
                e.currentTarget.style.boxShadow = `0 10px 25px -5px rgba(0, 0, 0, 0.5), 0 0 0 1px ${hexToRgba(colorTheme.buttonBadge, 0.1)}`;
              }}
            >
              <button
                type="button"
                onClick={() => toggleQuestion(index)}
                className="flex w-full items-center justify-between gap-4 p-3 text-left transition-colors hover:bg-slate-800/50 sm:p-4"
                aria-expanded={isOpen}
              >
                <h3 className="flex-1 text-sm font-semibold leading-tight text-white sm:text-base">
                  {item.question}
                </h3>
                <svg
                  className={`h-5 w-5 flex-shrink-0 transition-transform duration-300 ${
                    isOpen ? "rotate-180" : ""
                  }`}
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                  strokeWidth={2.5}
                  style={{ color: colorTheme.buttonBadge }}
                >
                  <path strokeLinecap="round" strokeLinejoin="round" d="M19 9l-7 7-7-7" />
                </svg>
              </button>
              <div
                className={`overflow-hidden transition-all duration-300 ${
                  isOpen ? "max-h-[500px] opacity-100" : "max-h-0 opacity-0"
                }`}
              >
                <div 
                  className="border-t p-3 sm:p-4"
                  style={{
                    borderColor: hexToRgba(colorTheme.buttonBadge, 0.5),
                    backgroundColor: hexToRgba(colorTheme.cardBackground, 0.3),
                  }}
                >
                  <p className="text-xs leading-relaxed text-slate-300/90 sm:text-sm sm:text-white/70">
                    {item.answer}
                  </p>
                </div>
              </div>
            </div>
          );
        })}
      </div>
    </section>
  );
}

